/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/

package rdc;

import java.net.URL;

import com.inet.report.Area;
import com.inet.report.BySummaryOrder;

import com.inet.report.Chart2;
import com.inet.report.DatabaseField;
import com.inet.report.Engine;
import com.inet.report.FieldElement;
import com.inet.report.Fields;
import com.inet.report.Group;
import com.inet.report.Section;
import com.inet.report.SummaryField;
import com.inet.report.chart.dataset.OneGroupDataset;
import com.inet.report.chart.plot.BarPlot;
import com.inet.report.chart.plot.BarStyle;

/**
 * This example explains how to add a chart to group header section of an existing report file. The chart shows top 5 of
 * customers sorted by "Last Year's Sales". <br>
 * <br>
 * You may to change the location of the report file that is used in this example. The report file is called mainrep.rpt
 * and comes with this RDC sample suite.
 * @see Chart2
 */
public class ChartTop5CustomerSample extends RDCSample {

    // Change this before running example,
    // if the url does not match the example report "mainrep.rpt"
    // comes with this RDC sample suite.
    private static final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Add a chart to group header section of an existing report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = new Engine( exportFmt );
            //set the name of the report file
            URL reportURL = getClass().getResource( '/' + MAINREPORT );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( MAINREPORT );
            }

            //too much data for this sample,  
            //with this selection formula data limited to only two countries 
            eng.setSF( "{Customers.Country} = \"USA\" OR  {Customers.Country} =\"Australia\" OR {Customers.Country} =\"Germany\"" );

            //get all fields defined in this report
            Fields dbFields = eng.getFields();

            //get two database fields for chart
            DatabaseField dbField0 = dbFields.getDatabaseField( "Customers.Country" );
            DatabaseField dbField1 = dbFields.getDatabaseField( "Customers.ContactName" );
            DatabaseField dbField2 = dbFields.getDatabaseField( "Customers.LastYearsSales" );

            //create a group
            Group groupCountry = eng.addGroup( dbField0 );

            //show group name
            Area area = groupCountry.getHeader();
            Section section = area.getSection( 0 );
            FieldElement groupName = section.addFieldElement( dbField0, 500, 100, 1500, 3000 );
            groupName.setItalic( true );
            groupName.setFontColor( 250 );
            groupName.setFontSize( 14 );

            //add chart and sets the type of chart (3D bar chart)
            Chart2 chart = section.addChart2( BarStyle.BAR3D, 2000, 100, 9000, 4000 );

            // Add a source to x axis
            BarPlot plot = (BarPlot)chart.getPlot();

            OneGroupDataset dataset = new OneGroupDataset( chart );
            dataset.setCategoryGroup( dbField1 );
            dataset.addDataField( SummaryField.SUM, dbField2, null, 0 );

            //gets the group for first x axis source to change sort order of data
            Group chartGroup = dataset.getCategoryGroup();

            //sets sort order to BySummary
            chartGroup.setSort( Group.SORT_SUMMARY_ORDER );

            //create and set a BySummaryOrder to this group, data will be sort by summary (TopN, BottomN)
            BySummaryOrder groupSort = new BySummaryOrder();
            chartGroup.setBySummaryOrder( groupSort );

            //set SummaryField by which data will be sorting 
            SummaryField sumField = dataset.getDataField( 0 );
            groupSort.addSummaryField( sumField );

            //set Top N
            groupSort.setSortType( BySummaryOrder.GROUP_SORT_TOP_N );

            //show only first 5 elements
            groupSort.setN( 5 );

            //discard all other elements
            groupSort.setOther( BySummaryOrder.OTHER_RECORDS_DISCARD );

            plot.setDataset( dataset );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new ChartTop5CustomerSample().initUI();
    }
}
